/**
 * @file
 * Defines behaviors for Braintree 3DS2 review checkout pane.
 */

(function ($, Drupal, drupalSettings, braintree) {

  'use strict';

  Drupal.behaviors.commerceBraintreeReview = {
    attach: function attach(context, settings) {
      var $form = $('form#' + settings.commerceBraintree.formId, context).once('braintree-attach');
      if ($form.length === 0) {
        return;
      }

      var waitForSdk = setInterval(function () {
        if (typeof braintree !== 'undefined') {
          var commerceBraintree = {};
          commerceBraintree = new Drupal.commerceBraintreeAuthorization($form, settings.commerceBraintree);
          $form.data('braintree', commerceBraintree);
          clearInterval(waitForSdk);
        }
      }, 100);

    },
    detach: function detach(context, settings, trigger) {
      if (trigger !== "unload") {
        return;
      }

      var $form = $('form#' + settings.commerceBraintree.formId, context).once('braintree-attach');
      if ($form.length === 0) {
        return;
      }
      var $submit = $form.find(':input.button--primary');
      var commerceBraintree = $form.data('braintree');
      // paypalCheckout doesn't have teardown() method.
      // See https://braintree.github.io/braintree-web/3.19.1/HostedFields.html
      // and https://braintree.github.io/braintree-web/3.19.1/PayPalCheckout.html
      if (commerceBraintree.integration.hasOwnProperty('teardown')) {
        commerceBraintree.integration.teardown();
      }
      $form.removeData('braintree');
      $form.removeOnce('braintree-attach');
      $form.off("submit.braintreeSubmit");
      $submit.prop('disabled', false);
    }
  };

  Drupal.commerceBraintreeAuthorization = function ($form, settings) {
    var $submit = $form.find(':input.button--primary');
    $form.append('<input type="hidden" name="_triggering_element_name" value="' + $submit.attr('name') + '" />');
    $form.append('<input type="hidden" name="_triggering_element_value" value="' + $submit.val() + '" />');
    var that = this;

    braintree.client.create({
      authorization: settings.clientToken
    }, function (clientError, clientInstance) {
      if (clientError) {
        console.error(clientError);
        return;
      }
      braintree.threeDSecure.create({
        // Use 3DS 2 when possible.
        version: 2,
        client: clientInstance,
      }, function (threeDSecureErr, threeDSecureInstance) {
        if (threeDSecureErr) {
          var message = that.errorMsg(threeDSecureErr);
          // Show the message above the form.
          $form.prepend(Drupal.theme('commerceBraintreeError', message));
          $('html, body').animate({
            scrollTop: $('[role="alert"').offset().top - 200
          }, 1000);
          return;
        }

        $submit.prop('disabled', false);

        $form.on('submit.braintreeSubmit', function (event, options) {
          options = options || {};
          if (options.tokenized) {
            // Tokenization complete, allow the form to submit.
            return;
          }

          event.preventDefault();
          $('.messages--error', $form).remove();

          threeDSecureInstance.verifyCard({
            amount: settings.amount,
            email: settings.email,
            nonce: settings.nonce,
            // Add in bin for 3DS2.
            bin: settings.bin,
            onLookupComplete: function (data, next) {
              // Use `data` here, then call `next()`.
              next();
            }
          }, function (verifyCardError, verification) {
            if (verifyCardError) {
              var message = that.errorMsg(verifyCardError);
              // Show the message above the form.
              $form.prepend(Drupal.theme('commerceBraintreeError', message));
              $('html, body').animate({
                scrollTop: $('[role="alert"]').offset().top - 200
              }, 1000);
              return;
            }
            $('.braintree-nonce', $form).val(verification.nonce);
            $form.trigger('submit', { 'tokenized' : true });
          });
        });
      });
    });
    return this;
  };

  Drupal.commerceBraintreeAuthorization.prototype.errorMsg = function (threeDSecureErr) {
    var message;

    switch (threeDSecureErr.code) {
      case 'THREEDS_NOT_ENABLED':
      case 'THREEDS_HTTPS_REQUIRED':
        message = Drupal.t('An error occurred while contacting the payment gateway.');
        break;
      case 'HOSTED_FIELDS_FIELDS_EMPTY':
        message = Drupal.t('Please enter your credit card details.');
        break;

      case 'HOSTED_FIELDS_FIELDS_INVALID':
        var fieldName = '';
        var fields = threeDSecureErr.details.invalidFieldKeys;
        if (fields.length > 0) {
          if (fields.length > 1) {
            var last = fields.pop();
            fieldName = fields.join(', ');
            fieldName += ' and ' + Drupal.t(last);
            message = Drupal.t('The @fields you entered are invalid.', {'@fields': fieldName});
          }
          else {
            fieldName = fields.pop();
            message = Drupal.t('The @field you entered is invalid.', {'@field': fieldName});
          }
        }
        else {
          message = Drupal.t('The payment details you entered are invalid.');
        }

        message += ' ' + Drupal.t('Please check your details and try again.');
        break;

      case 'HOSTED_FIELDS_TOKENIZATION_CVV_VERIFICATION_FAILED':
        message = Drupal.t('The CVV you entered is invalid.');
        message += ' ' + Drupal.t('Please check your details and try again.');
        break;

      case 'HOSTED_FIELDS_FAILED_TOKENIZATION':
        message = Drupal.t('An error occurred while contacting the payment gateway.');
        message += ' ' + Drupal.t('Please check your details and try again.');
        break;

      case 'HOSTED_FIELDS_TOKENIZATION_NETWORK_ERROR':
        message = Drupal.t('Could not connect to the payment gateway.');
        break;

      default:
        message = threeDSecureErr.message;
    }

    return message;
  };

})(jQuery, Drupal, drupalSettings, window.braintree);
